// Main popup script - Simplified version for purchases and downloads only
// API URL is hardcoded to https://api.unshared.shop
const API_BASE_URL = 'https://api.unshared.shop';
let settings = {};
let purchases = [];

// Initialize on load
document.addEventListener('DOMContentLoaded', async () => {
    await loadSettings();
    await checkAuthStatus();
    setupEventListeners();
});

async function loadSettings() {
    const result = await chrome.storage.local.get(['settings']);
    settings = result.settings || {
        apiToken: '',
        username: ''
    };
    
    // Populate form fields if we have saved username (but not password for security)
    if (settings.username) {
        document.getElementById('username').value = settings.username;
    }
}

async function checkAuthStatus() {
    if (settings.apiToken) {
        updateStatus('Authenticated', 'connected');
        document.getElementById('purchasesSection').style.display = 'block';
        document.getElementById('loginSection').style.display = 'none';
        await loadPurchases();
    } else {
        updateStatus('Please login', 'error');
        document.getElementById('purchasesSection').style.display = 'none';
        document.getElementById('loginSection').style.display = 'block';
    }
}

function setupEventListeners() {
    document.getElementById('loginBtn').addEventListener('click', handleLogin);
    document.getElementById('refreshBtn').addEventListener('click', loadPurchases);
    const logoutBtn = document.getElementById('logoutBtn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', () => {
            // Clear token and show login
            settings.apiToken = '';
            settings.username = '';
            chrome.storage.local.set({ settings });
            document.getElementById('loginSection').style.display = 'block';
            document.getElementById('purchasesSection').style.display = 'none';
            document.getElementById('username').value = '';
            document.getElementById('password').value = '';
            document.getElementById('username').focus();
            updateStatus('Logged out', 'error');
        });
    }
}

async function handleLogin() {
    const username = document.getElementById('username').value.trim();
    const password = document.getElementById('password').value;
    
    if (!username || !password) {
        updateStatus('Please enter username and password', 'error');
        return;
    }
    
    showLoading(true);
    updateStatus('Logging in...', 'connecting');
    
    try {
        const apiClient = new ApiClient(API_BASE_URL);
        const response = await apiClient.login(username, password);
        
        if (response.success && response.api_token) {
            // Save the API token from login response (same as C# app)
            settings.apiToken = response.api_token.trim().replace(/[\r\n\s]/g, '');
            settings.username = username; // Save username for convenience
            
            await chrome.storage.local.set({ settings });
            
            updateStatus('Login successful', 'connected');
            document.getElementById('purchasesSection').style.display = 'block';
            document.getElementById('loginSection').style.display = 'none';
            
            // Automatically load purchases after successful login
            await loadPurchases();
        } else {
            updateStatus(`Login failed: ${response.error || 'Unknown error'}`, 'error');
            document.getElementById('password').value = ''; // Clear password on error
        }
    } catch (error) {
        updateStatus(`Error: ${error.message}`, 'error');
        document.getElementById('password').value = ''; // Clear password on error
    } finally {
        showLoading(false);
    }
}

async function loadPurchases() {
    if (!settings.apiToken) {
        updateStatus('Please login first', 'error');
        document.getElementById('loginSection').style.display = 'block';
        document.getElementById('purchasesSection').style.display = 'none';
        return;
    }
    
    showLoading(true);
    updateStatus('Loading purchases...', 'connecting');
    
    try {
        const apiClient = new ApiClient(API_BASE_URL);
        apiClient.setApiToken(settings.apiToken);
        const response = await apiClient.getDownloads();
        
        if (response.success && response.purchases) {
            // Filter purchases to only show 1key, 2keys, 3keys, and 1keypluslifetime (case-insensitive)
            const allowedProducts = ['1key', '2keys', '3keys', '1keypluslifetime'];
            purchases = response.purchases.filter(purchase => {
                const productName = (purchase.product_name || '').toLowerCase();
                return allowedProducts.some(allowed => productName.includes(allowed.toLowerCase()));
            });
            displayPurchases();
            updateStatus(`Loaded ${purchases.length} purchase(s)`, 'connected');
        } else {
            const errorMsg = response.error || 'Unknown error';
            updateStatus(`Error: ${errorMsg}`, 'error');
            
            // If 401 error, show helpful message
            if (errorMsg.includes('401') || errorMsg.includes('Authentication')) {
                setTimeout(() => {
                    alert('Authentication Error!\n\n' +
                          'Please check that your API token is correct.\n\n' +
                          'You can get your API token from:\n' +
                          'https://unshared.shop/dashboard/myProfile.php\n\n' +
                          'Make sure to copy the entire token without any spaces or extra characters.');
                }, 500);
            }
            
            purchases = [];
            displayPurchases();
        }
    } catch (error) {
        updateStatus(`Error: ${error.message}`, 'error');
        purchases = [];
        displayPurchases();
    } finally {
        showLoading(false);
    }
}

function displayPurchases() {
    const container = document.getElementById('purchasesList');
    
    if (purchases.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-box-open"></i>
                <p>No purchases found. Click Refresh to reload.</p>
            </div>
        `;
        return;
    }
    
    container.innerHTML = purchases.map((purchase, index) => {
        const isKv = purchase.is_kv || (purchase.product_name || '').toLowerCase().includes('kv');
        const status = purchase.status || purchase.payment_status || 'N/A';
        const date = formatDate(purchase.purchase_date || purchase.payment_date);
        
        return `
        <div class="purchase-item">
            <div class="purchase-header">
                <div class="purchase-name">
                    ${escapeHtml(purchase.product_name || 'N/A')}
                    ${isKv ? '<span class="badge badge-kv">KV</span>' : ''}
                </div>
                <div class="purchase-date">${date}</div>
            </div>
            <div class="purchase-details">
                <span><i class="fas fa-receipt"></i> ${escapeHtml(purchase.transaction_id || 'N/A')}</span>
                <span><i class="fas fa-check-circle"></i> ${escapeHtml(status)}</span>
            </div>
            ${purchase.download_url ? `
                <button class="download-btn" onclick="downloadPurchase(${index})">
                    <i class="fas fa-download"></i> Download
                </button>
            ` : '<p style="color: var(--text-secondary); font-size: 12px; margin: 0;">No download available</p>'}
        </div>
    `}).join('');
}

async function downloadPurchase(index) {
    const purchase = purchases[index];
    if (!purchase || !purchase.download_url) {
        updateStatus('Invalid purchase or missing download URL', 'error');
        return;
    }
    
    const downloadBtn = event.target.closest('.download-btn');
    if (downloadBtn) {
        downloadBtn.disabled = true;
        downloadBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Downloading...';
    }
    
    updateStatus('Downloading file...', 'connecting');
    
    try {
        const apiClient = new ApiClient(API_BASE_URL);
        apiClient.setApiToken(settings.apiToken);
        
        // Download file
        const blob = await apiClient.downloadKvFile(purchase.download_url);
        
        // Get filename from URL or use product name
        const url = new URL(purchase.download_url, API_BASE_URL);
        const filename = purchase.product_name 
            ? `${purchase.product_name.replace(/[^a-z0-9]/gi, '_')}_${purchase.transaction_id}.zip`
            : `unshared_${purchase.transaction_id}.zip`;
        
        // Create download link
        const url_obj = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url_obj;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url_obj);
        
        updateStatus('Download started!', 'connected');
        
        // Use Chrome downloads API as fallback
        try {
            const blobUrl = await blobToDataURL(blob);
            chrome.downloads.download({
                url: blobUrl,
                filename: filename,
                saveAs: false
            });
        } catch (e) {
            // Fallback method already triggered above
        }
        
        setTimeout(() => {
            if (downloadBtn) {
                downloadBtn.disabled = false;
                downloadBtn.innerHTML = '<i class="fas fa-download"></i> Download';
            }
            updateStatus(`Loaded ${purchases.length} purchase(s)`, 'connected');
        }, 2000);
        
    } catch (error) {
        updateStatus(`Download failed: ${error.message}`, 'error');
        if (downloadBtn) {
            downloadBtn.disabled = false;
            downloadBtn.innerHTML = '<i class="fas fa-download"></i> Download';
        }
        alert(`Download failed: ${error.message}`);
    }
}

function blobToDataURL(blob) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
    });
}

function updateStatus(message, type = '') {
    const statusText = document.getElementById('statusText');
    const indicator = document.getElementById('connectionIndicator');
    
    statusText.textContent = message;
    
    indicator.className = 'connection-indicator';
    if (type === 'connected' || type === 'success') {
        indicator.classList.add('connected');
    } else if (type === 'connecting') {
        indicator.classList.add('connecting');
    } else if (type === 'error') {
        indicator.classList.add('error');
    }
}

function showLoading(show) {
    document.getElementById('loadingOverlay').style.display = show ? 'flex' : 'none';
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    try {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
    } catch {
        return dateString;
    }
}

// Make downloadPurchase available globally
window.downloadPurchase = downloadPurchase;