// API Client for Unshared API
class ApiClient {
    constructor(baseUrl) {
        this.baseUrl = baseUrl.replace(/\/+$/, '');
        this.apiToken = null;
    }

    setApiToken(token) {
        this.apiToken = token ? token.trim().replace(/[\r\n\s]/g, '') : null;
    }

    async login(username, password) {
        try {
            const formData = new URLSearchParams();
            formData.append('action', 'login');
            formData.append('username', username);
            formData.append('password', password);

            const response = await fetch(`${this.baseUrl}/?action=login`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded'
                },
                body: formData
            });

            const responseText = await response.text();
            
            // Check if response is HTML (error page)
            if (responseText.trim().startsWith('<')) {
                return {
                    success: false,
                    error: `Server returned HTML instead of JSON. Status: ${response.status}`
                };
            }

            const data = JSON.parse(responseText);
            return data;
        } catch (error) {
            return {
                success: false,
                error: error.message
            };
        }
    }

    async getProfile() {
        if (!this.apiToken) {
            return { success: false, error: 'No API token set' };
        }

        try {
            const url = `${this.baseUrl}/?action=profile&api_token=${encodeURIComponent(this.apiToken)}`;
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${this.apiToken}`,
                    'X-API-Token': this.apiToken
                }
            });

            const responseText = await response.text();
            
            if (responseText.trim().startsWith('<')) {
                return {
                    success: false,
                    error: `Server returned HTML instead of JSON. Status: ${response.status}`
                };
            }

            const data = JSON.parse(responseText);
            return data;
        } catch (error) {
            return {
                success: false,
                error: error.message
            };
        }
    }

    async getDownloads() {
        if (!this.apiToken) {
            return { success: false, error: 'No API token set' };
        }

        try {
            const url = `${this.baseUrl}/?action=downloads&api_token=${encodeURIComponent(this.apiToken)}`;
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${this.apiToken}`,
                    'X-API-Token': this.apiToken
                }
            });

            const responseText = await response.text();
            
            if (responseText.trim().startsWith('<')) {
                return {
                    success: false,
                    error: `Server returned HTML instead of JSON. Status: ${response.status}`
                };
            }

            if (!response.ok) {
                let errorMsg = response.statusText || 'Unknown error';
                let errorDetails = '';
                try {
                    const errorData = JSON.parse(responseText);
                    errorMsg = errorData.error || errorData.message || errorMsg;
                    errorDetails = errorData.error || '';
                } catch {}
                
                // Provide helpful error message for 401
                if (response.status === 401) {
                    errorMsg = 'Authentication failed. Please check that your API token is correct and not expired. You can get your API token from: https://unshared.shop/dashboard/myProfile.php';
                }
                
                return {
                    success: false,
                    error: `HTTP ${response.status}: ${errorMsg}${errorDetails ? ' - ' + errorDetails : ''}`
                };
            }

            const data = JSON.parse(responseText);
            
            // Process purchases - mark KV purchases
            if (data.success && data.purchases) {
                data.purchases.forEach(purchase => {
                    if (!purchase.is_kv) {
                        const productNameLower = (purchase.product_name || '').toLowerCase();
                        purchase.is_kv = productNameLower.includes('1kv') ||
                                        productNameLower.includes('1key') ||
                                        productNameLower.includes('kv') ||
                                        productNameLower.includes('keyvault') ||
                                        productNameLower.includes('key vault');
                    }
                    
                    // Fix download URL if relative
                    if (purchase.download_url && !purchase.download_url.startsWith('http')) {
                        purchase.download_url = this.baseUrl + purchase.download_url.replace(/^\/api\//, '/');
                    }
                });
            }

            return data;
        } catch (error) {
            return {
                success: false,
                error: error.message
            };
        }
    }

    async downloadKvFile(downloadUrl) {
        if (!this.apiToken) {
            throw new Error('No API token set');
        }

        // Handle both absolute and relative URLs
        let fullUrl = downloadUrl;
        if (!downloadUrl.startsWith('http')) {
            fullUrl = this.baseUrl + downloadUrl.replace(/^\/api\//, '/');
        }

        const response = await fetch(fullUrl, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${this.apiToken}`,
                'X-API-Token': this.apiToken
            }
        });

        if (!response.ok) {
            throw new Error(`Download failed: ${response.status} ${response.statusText}`);
        }

        return await response.blob();
    }
}
